# 🚀 VYLO Deployment Guide - RedHat Server with Node.js

## Prerequisites on RedHat Server

1. **SSH Access** to your RedHat server
2. **Root or sudo privileges**
3. **Domain name** pointed to your server IP (optional but recommended)

---

## Step 1: Install Node.js on RedHat

```bash
# SSH into your RedHat server
ssh user@your-server-ip

# Install Node.js 20.x (LTS)
sudo dnf module enable nodejs:20
sudo dnf install nodejs -y

# Verify installation
node --version  # Should show v20.x.x
npm --version   # Should show 10.x.x
```

---

## Step 2: Install PM2 (Process Manager)

```bash
# Install PM2 globally
sudo npm install -g pm2

# Verify
pm2 --version
```

---

## Step 3: Install and Configure Nginx

```bash
# Install Nginx
sudo dnf install nginx -y

# Start and enable Nginx
sudo systemctl start nginx
sudo systemctl enable nginx

# Check status
sudo systemctl status nginx
```

---

## Step 4: Upload VYLO to Server

### Option A: Using Git (Recommended)

```bash
# On your RedHat server
cd /var/www
sudo mkdir vylo
sudo chown $USER:$USER vylo
cd vylo

# Clone your repository (if you have it on GitHub)
git clone https://github.com/yourusername/vylo.git .

# Or initialize git and push from your local machine
```

### Option B: Using SCP/SFTP

```bash
# From your Windows machine (in PowerShell)
cd C:\Users\Digit\source\repos\vylo

# Create production build first
npm run build

# Upload to server (adjust path and user)
scp -r ./* user@your-server-ip:/var/www/vylo/
```

---

## Step 5: Install Dependencies & Build on Server

```bash
# On your RedHat server
cd /var/www/vylo

# Install dependencies
npm install --production

# Build the application
npm run build
```

---

## Step 6: Configure PM2

```bash
# Create PM2 ecosystem file
cat > ecosystem.config.js << 'EOF'
module.exports = {
  apps: [{
    name: 'vylo',
    script: 'npm',
    args: 'start',
    cwd: '/var/www/vylo',
    instances: 'max',
    exec_mode: 'cluster',
    watch: false,
    max_memory_restart: '1G',
    env: {
      NODE_ENV: 'production',
      PORT: 3000
    }
  }]
}
EOF

# Start VYLO with PM2
pm2 start ecosystem.config.js

# Save PM2 configuration
pm2 save

# Setup PM2 to start on boot
pm2 startup systemd
# Follow the command it outputs
```

---

## Step 7: Configure Nginx Reverse Proxy

```bash
# Create Nginx configuration
sudo nano /etc/nginx/conf.d/vylo.conf
```

Add this configuration:

```nginx
server {
    listen 80;
    server_name your-domain.com www.your-domain.com;  # Replace with your domain

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}
```

```bash
# Test Nginx configuration
sudo nginx -t

# Reload Nginx
sudo systemctl reload nginx
```

---

## Step 8: Configure Firewall

```bash
# Allow HTTP and HTTPS
sudo firewall-cmd --permanent --add-service=http
sudo firewall-cmd --permanent --add-service=https
sudo firewall-cmd --reload

# Verify
sudo firewall-cmd --list-all
```

---

## Step 9: Setup SSL with Let's Encrypt (Optional but Recommended)

```bash
# Install Certbot
sudo dnf install certbot python3-certbot-nginx -y

# Get SSL certificate
sudo certbot --nginx -d your-domain.com -d www.your-domain.com

# Certbot will automatically configure Nginx for HTTPS
# Certificates auto-renew via cron
```

---

## Step 10: Environment Variables

```bash
# Create .env.local file
cd /var/www/vylo
nano .env.local
```

Add your environment variables:

```env
# Database
DATABASE_URL="postgresql://user:password@localhost:5432/vylo"

# AI API Keys
ANTHROPIC_API_KEY="your-claude-api-key"
OPENAI_API_KEY="your-openai-api-key"

# Authentication
NEXTAUTH_URL="https://your-domain.com"
NEXTAUTH_SECRET="generate-a-secret-key-here"

# Stripe (for payments)
STRIPE_SECRET_KEY="your-stripe-secret-key"
STRIPE_PUBLISHABLE_KEY="your-stripe-publishable-key"
```

```bash
# Restart PM2 to apply changes
pm2 restart vylo
```

---

## Useful PM2 Commands

```bash
# View logs
pm2 logs vylo

# Monitor
pm2 monit

# Restart
pm2 restart vylo

# Stop
pm2 stop vylo

# View status
pm2 status

# View detailed info
pm2 info vylo
```

---

## Updating VYLO

```bash
# Pull latest changes
cd /var/www/vylo
git pull

# Install dependencies
npm install --production

# Rebuild
npm run build

# Restart
pm2 restart vylo
```

---

## Database Setup (PostgreSQL)

```bash
# Install PostgreSQL
sudo dnf install postgresql-server postgresql-contrib -y

# Initialize database
sudo postgresql-setup --initdb

# Start PostgreSQL
sudo systemctl start postgresql
sudo systemctl enable postgresql

# Create database and user
sudo -u postgres psql

# In PostgreSQL prompt:
CREATE DATABASE vylo;
CREATE USER vylouser WITH PASSWORD 'your-secure-password';
GRANT ALL PRIVILEGES ON DATABASE vylo TO vylouser;
\q

# Run Prisma migrations
cd /var/www/vylo
npx prisma migrate deploy
```

---

## Troubleshooting

### Check if Next.js is running
```bash
curl http://localhost:3000
```

### Check Nginx error logs
```bash
sudo tail -f /var/log/nginx/error.log
```

### Check PM2 logs
```bash
pm2 logs vylo --lines 100
```

### Restart everything
```bash
pm2 restart vylo
sudo systemctl restart nginx
```

---

## Security Checklist

- [ ] Firewall configured (only allow 22, 80, 443)
- [ ] SSL certificate installed
- [ ] Strong database password
- [ ] Environment variables secured
- [ ] Regular backups scheduled
- [ ] Server updates automated
- [ ] PM2 startup configured
- [ ] Rate limiting configured in Nginx

---

## Performance Optimization

Add to `/etc/nginx/nginx.conf` in http block:

```nginx
# Gzip compression
gzip on;
gzip_vary on;
gzip_types text/plain text/css text/xml text/javascript application/x-javascript application/xml+rss application/json;

# Cache static files
location /_next/static {
    alias /var/www/vylo/.next/static;
    expires 365d;
    access_log off;
}
```

---

**You're all set!** VYLO will be live at `https://your-domain.com` 🚀

For issues, check:
1. PM2 logs: `pm2 logs vylo`
2. Nginx logs: `sudo tail -f /var/log/nginx/error.log`
3. System logs: `sudo journalctl -xe`
