"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";

export default function CreateProjectPage() {
  const router = useRouter();
  const [name, setName] = useState("");
  const [description, setDescription] = useState("");
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      const res = await fetch("/api/projects", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          name,
          description,
          userId: "demo-user", // TODO: Get from auth
        }),
      });

      if (res.ok) {
        const { project } = await res.json();
        router.push(`/dashboard/projects/${project.id}`);
      } else {
        alert("Failed to create project");
      }
    } catch (error) {
      alert("Error creating project");
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-vylo-darker via-vylo-dark to-vylo-darker">
      <header className="border-b border-vylo-accent/20 bg-vylo-dark/50 backdrop-blur-xl">
        <div className="container mx-auto px-6 py-4">
          <Link href="/dashboard" className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-gradient-to-br from-vylo-accent to-vylo-primary vylo-glow"></div>
            <h1 className="text-2xl font-black vylo-text-gradient">VYLO</h1>
          </Link>
        </div>
      </header>

      <div className="container mx-auto px-6 py-12">
        <div className="max-w-2xl mx-auto">
          <h2 className="text-4xl font-black mb-2">Create New Project</h2>
          <p className="text-gray-400 mb-8">
            Let's bring your idea to life! Vee and the AI agents are ready to help 🚀
          </p>

          <form onSubmit={handleSubmit} className="vylo-card space-y-6">
            <div>
              <label className="block text-sm font-semibold mb-2">
                Project Name <span className="text-vylo-error">*</span>
              </label>
              <input
                type="text"
                value={name}
                onChange={(e) => setName(e.target.value)}
                className="vylo-input w-full"
                placeholder="My Awesome App"
                required
                maxLength={100}
              />
            </div>

            <div>
              <label className="block text-sm font-semibold mb-2">
                Description
              </label>
              <textarea
                value={description}
                onChange={(e) => setDescription(e.target.value)}
                className="vylo-input w-full min-h-[120px]"
                placeholder="Describe what you want to build..."
                maxLength={500}
              />
              <p className="text-xs text-gray-500 mt-1">
                {description.length}/500 characters
              </p>
            </div>

            <div className="vylo-card bg-vylo-darker/50 border-vylo-accent/10">
              <h4 className="font-semibold mb-3">What Gets Created:</h4>
              <ul className="space-y-2 text-sm text-gray-400">
                <li className="flex items-center gap-2">
                  <span className="text-vylo-primary">✓</span>
                  8 Project Sections (Logo, Icons, Graphics, Marketing, Wireframe, Code, Testing, Deployment)
                </li>
                <li className="flex items-center gap-2">
                  <span className="text-vylo-primary">✓</span>
                  3 AI Agents (Claude Code Master, GPT-4 Writer, DALL-E Designer)
                </li>
                <li className="flex items-center gap-2">
                  <span className="text-vylo-primary">✓</span>
                  Automated task management & progress tracking
                </li>
              </ul>
            </div>

            <div className="flex gap-4">
              <button
                type="submit"
                disabled={loading || !name}
                className="vylo-button flex-1 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {loading ? "Creating..." : "Create Project 🚀"}
              </button>
              <Link
                href="/dashboard"
                className="px-6 py-3 rounded-lg border border-vylo-accent/30 hover:border-vylo-primary transition-all text-center"
              >
                Cancel
              </Link>
            </div>
          </form>
        </div>
      </div>
    </div>
  );
}
