"use client";

import { useState } from "react";
import Link from "next/link";
import type { Project, AIAgent } from "@/types";
import UserMenu from "@/components/UserMenu";

// Mock data for demo
const mockProjects: Project[] = [
  {
    id: "1",
    name: "E-Commerce Platform",
    description: "Full-stack e-commerce with AI recommendations",
    status: "developing",
    progress: 65,
    createdAt: new Date(),
    updatedAt: new Date(),
    owner: "user-1",
    aiAgents: [],
    sections: [],
  },
  {
    id: "2",
    name: "Mobile Fitness App",
    description: "Health tracking with social features",
    status: "designing",
    progress: 40,
    createdAt: new Date(),
    updatedAt: new Date(),
    owner: "user-1",
    aiAgents: [],
    sections: [],
  },
];

const mockAgents: AIAgent[] = [
  {
    id: "agent-1",
    name: "Claude Code Master",
    type: "code",
    status: "working",
    currentTask: "Building authentication system",
    provider: "claude",
    progress: 78,
    lastActivity: new Date(),
  },
  {
    id: "agent-2",
    name: "GPT-4 Content Writer",
    type: "marketing",
    status: "working",
    currentTask: "Creating landing page copy",
    provider: "gpt4",
    progress: 92,
    lastActivity: new Date(),
  },
  {
    id: "agent-3",
    name: "DALL-E Designer",
    type: "design",
    status: "idle",
    currentTask: undefined,
    provider: "dalle",
    progress: 0,
    lastActivity: new Date(),
  },
];

export default function DashboardPage() {
  const [selectedProject, setSelectedProject] = useState<string | null>(null);

  const getStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      planning: "bg-gray-500",
      designing: "bg-purple-500",
      developing: "bg-blue-500",
      testing: "bg-yellow-500",
      deploying: "bg-orange-500",
      deployed: "bg-green-500",
    };
    return colors[status] || "bg-gray-500";
  };

  const getAgentStatusColor = (status: string) => {
    const colors: Record<string, string> = {
      idle: "bg-gray-500",
      working: "bg-vylo-primary",
      completed: "bg-vylo-success",
      error: "bg-vylo-error",
    };
    return colors[status] || "bg-gray-500";
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-vylo-darker via-vylo-dark to-vylo-darker">
      {/* Header */}
      <header className="border-b border-vylo-accent/20 bg-vylo-dark/50 backdrop-blur-xl sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex justify-between items-center">
            <Link href="/" className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-full bg-gradient-to-br from-vylo-accent to-vylo-primary vylo-glow"></div>
              <h1 className="text-2xl font-black vylo-text-gradient">VYLO</h1>
            </Link>

            <nav className="flex items-center gap-6">
              <Link href="/dashboard" className="text-vylo-primary font-semibold">
                Dashboard
              </Link>
              <Link href="/dashboard/agents" className="text-gray-400 hover:text-white transition">
                AI Agents
              </Link>
              <Link href="/dashboard/settings" className="text-gray-400 hover:text-white transition">
                Settings
              </Link>
              <UserMenu />
            </nav>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-6 py-8">
        {/* Welcome Section */}
        <div className="mb-8">
          <h2 className="text-4xl font-black mb-2">Welcome back! 👋</h2>
          <p className="text-gray-400 text-lg">
            Your AI agents are working hard. <span className="text-vylo-primary">{mockAgents.filter(a => a.status === "working").length} active right now</span>
          </p>
        </div>

        {/* Quick Actions */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-8">
          <Link href="/dashboard/create-project" className="vylo-button flex items-center justify-center gap-2">
            <span className="text-2xl">➕</span> New Project
          </Link>
          <button className="bg-vylo-dark/50 border border-vylo-accent/30 hover:border-vylo-primary px-6 py-3 rounded-lg transition-all">
            <span className="text-2xl">🤖</span> Add AI Agent
          </button>
          <button className="bg-vylo-dark/50 border border-vylo-accent/30 hover:border-vylo-primary px-6 py-3 rounded-lg transition-all">
            <span className="text-2xl">📊</span> View Analytics
          </button>
          <button className="bg-vylo-dark/50 border border-vylo-accent/30 hover:border-vylo-primary px-6 py-3 rounded-lg transition-all">
            <span className="text-2xl">🚀</span> Deploy Now
          </button>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
          {/* Projects Column */}
          <div className="lg:col-span-2 space-y-6">
            <div className="flex justify-between items-center">
              <h3 className="text-2xl font-bold">Active Projects</h3>
              <button className="text-vylo-primary hover:text-vylo-secondary transition">
                View All →
              </button>
            </div>

            {mockProjects.map((project) => (
              <Link
                key={project.id}
                href={`/dashboard/projects/${project.id}`}
                className="vylo-card cursor-pointer hover:scale-[1.02] block"
              >
                <div className="flex justify-between items-start mb-4">
                  <div>
                    <h4 className="text-xl font-bold mb-1">{project.name}</h4>
                    <p className="text-gray-400 text-sm">{project.description}</p>
                  </div>
                  <span className={`px-3 py-1 rounded-full text-xs font-semibold ${getStatusColor(project.status)}`}>
                    {project.status}
                  </span>
                </div>

                {/* Progress Bar */}
                <div className="mb-4">
                  <div className="flex justify-between text-sm mb-2">
                    <span className="text-gray-400">Progress</span>
                    <span className="text-vylo-primary font-semibold">{project.progress}%</span>
                  </div>
                  <div className="h-2 bg-vylo-darker rounded-full overflow-hidden">
                    <div
                      className="h-full bg-gradient-to-r from-vylo-accent to-vylo-primary transition-all duration-500"
                      style={{ width: `${project.progress}%` }}
                    ></div>
                  </div>
                </div>

                {/* Quick Stats */}
                <div className="flex gap-4 text-sm">
                  <div className="flex items-center gap-1">
                    <span className="text-vylo-success">●</span>
                    <span className="text-gray-400">2 agents active</span>
                  </div>
                  <div className="flex items-center gap-1">
                    <span className="text-vylo-warning">●</span>
                    <span className="text-gray-400">5 tasks pending</span>
                  </div>
                </div>
              </Link>
            ))}
          </div>

          {/* AI Agents Column */}
          <div className="space-y-6">
            <h3 className="text-2xl font-bold">AI Agents</h3>

            {mockAgents.map((agent) => (
              <div key={agent.id} className="vylo-card">
                <div className="flex items-start gap-4">
                  {/* Agent Avatar */}
                  <div className={`w-12 h-12 rounded-full ${getAgentStatusColor(agent.status)} vylo-glow flex items-center justify-center text-2xl`}>
                    {agent.type === "code" && "💻"}
                    {agent.type === "marketing" && "📝"}
                    {agent.type === "design" && "🎨"}
                  </div>

                  <div className="flex-1">
                    <div className="flex justify-between items-start mb-2">
                      <h4 className="font-semibold">{agent.name}</h4>
                      <span className={`w-2 h-2 rounded-full ${getAgentStatusColor(agent.status)} animate-pulse`}></span>
                    </div>

                    {agent.currentTask && (
                      <p className="text-sm text-gray-400 mb-2">{agent.currentTask}</p>
                    )}

                    {agent.status === "working" && (
                      <div className="h-1 bg-vylo-darker rounded-full overflow-hidden">
                        <div
                          className="h-full bg-vylo-primary animate-pulse"
                          style={{ width: `${agent.progress}%` }}
                        ></div>
                      </div>
                    )}

                    {agent.status === "idle" && (
                      <p className="text-xs text-gray-500">Waiting for tasks...</p>
                    )}
                  </div>
                </div>
              </div>
            ))}

            {/* System Stats */}
            <div className="vylo-card">
              <h4 className="font-semibold mb-4">System Status</h4>
              <div className="space-y-3">
                <div className="flex justify-between">
                  <span className="text-gray-400 text-sm">Total Projects</span>
                  <span className="font-semibold">2</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400 text-sm">Active Agents</span>
                  <span className="font-semibold text-vylo-primary">2</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400 text-sm">Completed Tasks</span>
                  <span className="font-semibold text-vylo-success">47</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-400 text-sm">AI Credits</span>
                  <span className="font-semibold">1,250</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
